#include "cliques.h"

/* Legacy note:

   This is the version of the MaxClique softvare written by Kevin O'Neill (read
   the comment bellow), modified by Zoran Stanic and Nedeljko Stefanovic.
*/

/* MaxClique - for finding maximal cliques and independent sets of graphs
 * Written by Kevin O'Neill
 * Based on algorithm of Tsukiyama, Ide, Ariyoshi, and Shirakawa
 * Latest version: December 18, 2003
 *
 * Copyright (C) 2003  Kevin O'Neill
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * The author can be contacted at oneill@cs.cornell.edu, or by mail at
 *   Computer Science Department
 *   4130 Upson Hall
 *   Cornell University
 *   Ithaca, NY 14853-7501
 */


/**
 * Standard includes
 */

#include <iostream>
#include <fstream>
#include <vector>
#include <stdlib.h>

using namespace std;

/**
 * Internal constants
 */

#define MAX_MATRIX_SIZE 500
#define CLIQUE_MODE 0
#define IS_MODE 1
#define PROGRAM_NAME "maxclique"

/**
 * Main data structures
 */

int matrix_size;
int current_mode;
vector< vector<int> > adjacency_matrix;
vector< vector<int> > adjacency_lists;
vector<int> num_adjacencies;
// Note: Bucket is a bit (0/1) array on the elements
// of the adjacency list for each node
vector< vector<int> > Bucket;
vector<int> IS;

/**
 * Forward declarations
 */

void load_matrix(istream&, float, bool = false);
void initialize_adjacency_lists_and_stuff();
void print_matrix();
void print_adjacency_lists();
void backtrack(int, ostream&);
void output_IS(ostream&);

/**
 * Main method
 */

bool expected;

void calculate_cliques(CalculatingWindow *w, istream &input, ostream &output, bool is_expected)
{
    float cutoff = 0;

    expected = is_expected;
	adjacency_matrix = vector< vector<int> >();
	adjacency_lists = vector< vector<int> >();
	num_adjacencies = vector<int>();

	if (w!=0)
        w->print("Maximal cliques computing...");

	load_matrix(input, cutoff);
	initialize_adjacency_lists_and_stuff();
	backtrack(0, output);

	if (w!=0)
        w->print("Maximal cliques computed.");
}

void load_matrix(istream &istr, float cutoff, bool is_mode)
{
	int i, j;
	float in_float;
	char c;

	current_mode = CLIQUE_MODE;

	if (is_mode)
		current_mode = IS_MODE;
	do
	{
	    CHECK_STOPPING
		c = istr.get();

		if (istr.eof())
            if (expected)
                throw("Warning: Illegal entries!");
            else
                throw("There is no extensions for given data!");
	}
	while(c!='0');

	adjacency_matrix.push_back(vector<int>());
	adjacency_matrix[0].push_back(0);

	do
	{
        CHECK_STOPPING
		c = istr.get();

		if (istr.eof())
            break;

		if (c>='0' && c<='1')
			adjacency_matrix[0].push_back(c-'0');

        else
            if (c!=' ' && c!='\t' && c!='\n')
                throw("Warning: Illegal entries!");
	}
	while (!istr.eof() && c!='\n');

	matrix_size = adjacency_matrix[0].size();

	for (i=0; i<matrix_size; i++)
	{
		adjacency_lists.push_back(vector<int>());
		Bucket.push_back(vector<int>());

		for (j=0; j<matrix_size; j++)
		{
            CHECK_STOPPING
			adjacency_lists[i].push_back(0);
			Bucket[i].push_back(0);
			num_adjacencies.push_back(0);
			IS.push_back(0);
		}
	}

	for (i=1; i<matrix_size; i++)
	{
		adjacency_matrix.push_back(vector<int>());

		for (j=0; j<matrix_size; j++)
		{
            CHECK_STOPPING
			istr >> c;

			if (istr.eof())
				throw("Warning: Illegal entries!");

            c -= '0';
			adjacency_matrix[i].push_back(c);

			if (i==j && c!=0)
				throw ("Warning: Illegal entries!");

			if (c!=0 && c!=1)
				throw ("Warning: Illegal entries!");

			if (i>j && c!=adjacency_matrix[j][i])
				throw ("Warning: Illegal entries!");
		}
	}

	istr >> c;

	if (!istr.eof())
		throw("Warning: Illegal entries!");
}

/**
 * Initialize adjacency lists, set all buckets to be empty, and set all IS values to 0
 *
 * Important note: Because the "backtrack" method finds independent sets, not cliques,
 * the algorithm needs to run on the "opposite/inverse" of the adjacency matrix. Thus the
 * "current_mode" (which inverts the graph when it's CLIQUE_MODE).
 */

void initialize_adjacency_lists_and_stuff()
{
	int i,j;

	for(i=0;i<matrix_size;i++)
	{
		for(j=i+1;j<matrix_size;j++)
		{
			// it's i+1, not i, because we ignore self-loops in the graph

			CHECK_STOPPING

			if (adjacency_matrix[i][j] == current_mode)
			{
				adjacency_lists[i][num_adjacencies[i]] = j;
				Bucket[i][num_adjacencies[i]] = 0;
				adjacency_lists[j][num_adjacencies[j]] = i;
				Bucket[j][num_adjacencies[j]] = 0;
				num_adjacencies[i]++;
				num_adjacencies[j]++;
      		}
    	}
  	}
}

/**
 * For testing, to make sure the adjacency matrix looks like it does in the file
 */

void print_matrix(ostream &ostr)
{
	int i,j;

	for(i=0;i<matrix_size;i++)
	{
	    CHECK_STOPPING

		for(j=0;j<i;j++)
			ostr << "  ";

    	for(j=i;j<matrix_size;j++)
      		ostr << adjacency_matrix[i][j] << " ";

    	ostr << "\n";
  	}
}


/**
 * Ditto for the adjacency lists
 */

void print_adjacency_lists(ostream &ostr)
{
	int i,j;

	for (i=0; i<matrix_size; i++)
	{
		ostr << i << " ";

		for(j=0; j<num_adjacencies[i]; j++)
		{
		    CHECK_STOPPING
			ostr << adjacency_lists[i][j] << " ";
    	}

    	ostr << endl;
  }
}


/**
 * This is the main method, taken directly from Tsukiyama et. al.
 */

void backtrack(int i, ostream &ostr)
{
	int c, x;
	int f;

	int adjIndexY, y;
	int adjIndexZ, z;
	int bucketIndex;

	if (i >= (matrix_size-1))
	{
    	// Output new MIS designated by IS( )
    	output_IS(ostr);
	}
	else
	{
    	x = i+1;
    	c = 0;

	    // for y \in Adj(x) such that y \leq i
	    for (adjIndexY=0; adjIndexY<num_adjacencies[x] && ((y=adjacency_lists[x][adjIndexY]) <= i); adjIndexY++)
		{
		    CHECK_STOPPING
            if (IS[y] == 0) c++;
	    }

	    if (c == 0)
		{
			// for y \in Adj(x) such that y \leq i
			for (adjIndexY=0; adjIndexY<num_adjacencies[x] && ((y=adjacency_lists[x][adjIndexY]) <= i); adjIndexY++)
			{
			    CHECK_STOPPING
				IS[y]++;
	      	}

	      	backtrack(x, ostr);
	      	// for y \in Adj(x) such that y \leq i
	      	for (adjIndexY=0; adjIndexY<num_adjacencies[x] && ((y=adjacency_lists[x][adjIndexY]) <= i); adjIndexY++)
			{
			    CHECK_STOPPING
				IS[y]--;
	      	}
		}
		else
		{
	    	IS[x] = c;
	    	backtrack(x, ostr);
	    	IS[x] = 0;
	    	f = TRUE;
	    	for (adjIndexY=0; adjIndexY<num_adjacencies[x] && ((y=adjacency_lists[x][adjIndexY]) <= i); adjIndexY++)
			{
				if (IS[y] == 0)
				{
					// Put y in Bucket(x):
					Bucket[x][adjIndexY] = 1;
					// for z \in Adj(y) such that z \leq i
					for (adjIndexZ=0; adjIndexZ<num_adjacencies[y] && ((z=adjacency_lists[y][adjIndexZ]) <= i); adjIndexZ++)
					{
					    CHECK_STOPPING
		    			IS[z]--;

						if(IS[z] == 0)
						{
							f = FALSE;
		    			}
		  			}
				}

				IS[y]++;
			}

			if (f)
			{
				backtrack(x, ostr);
	      	}

			for (adjIndexY=0; adjIndexY<num_adjacencies[x] && ((y=adjacency_lists[x][adjIndexY]) <= i); adjIndexY++)
			{
			    CHECK_STOPPING
				IS[y]--;
	      	}

	      	// for y \in Bucket(x) do
	      	for (adjIndexY=0; adjIndexY<num_adjacencies[x]; adjIndexY++)
			{
				if (Bucket[x][adjIndexY])
				{
		  			y = adjacency_lists[x][adjIndexY];

		  			for (adjIndexZ=0; adjIndexZ<num_adjacencies[y] && ((z=adjacency_lists[y][adjIndexZ]) <= i); adjIndexZ++)
					{
					    CHECK_STOPPING
		    			IS[z]++;
		  			}

		  			// delete y from Bucket(x)
		  			Bucket[x][adjIndexY] = 0;
				}
	      	}
	    }
  	}
}

void output_IS(ostream &ostr)
{
	int ISIndex;

	if (current_mode == CLIQUE_MODE)
    	ostr << "Maximal clique: ";
    else
    	ostr << "Maximal independent set: ";

    for(ISIndex=0;ISIndex<matrix_size;ISIndex++)
	{
	    CHECK_STOPPING
    	// IS[v] = 0 iff v is in MIS (see paper)
    	if (IS[ISIndex] == 0)
			ostr << ISIndex+1 << " ";
    	// ISIndex+1 so that node numbering starts with 1
    }

    ostr << endl;
}

void maximal_cliques(CalculatingWindow &w, const char *input_file_name,
                                           const char *output_file_name)
{
    ifstream input_file(input_file_name);

    if (!input_file)
        throw("Warning: No input file.");

    ofstream output_file(output_file_name);
    calculate_cliques(&w, input_file, output_file);
}
